-- MODAVA Database Schema
-- This file contains all the necessary tables for the MODAVA fashion SaaS platform

-- Enable UUID extension
CREATE EXTENSION IF NOT EXISTS "uuid-ossp";

-- Users table (extends Supabase auth.users)
-- This is handled by Supabase Auth automatically
-- You can extend it with additional fields if needed

-- Plans Table
create table plans (
  id uuid primary key default gen_random_uuid(),
  name text not null,
  price numeric not null,
  trial_period_days integer default 0,
  created_at timestamp with time zone default timezone('utc', now())
);

create table admin_settings (
  id uuid primary key default gen_random_uuid(),
  payout_threshold_amount numeric default 0,
  kyc_required_above_threshold boolean default true,
  contributor_discount_rules jsonb,
  referral_bonus_rules jsonb,
  commission_settings jsonb,
  created_at timestamp with time zone default timezone('utc', now())
);


create table vendors (
  id uuid primary key default gen_random_uuid(),
  user_id uuid unique not null, -- Supabase Auth UID
  business_name text,
  phone_number text,
  email text,
  onboarding_status text default 'incomplete',
  store_link text unique,
  plan_id uuid references plans(id),
  kyc_status text default 'pending',
  referral_code text,
  feature_flags jsonb,
  trial_end_date date,
  created_at timestamp with time zone default timezone('utc', now())
);



create table customers (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  name text not null,
  phone text,
  email text,
  notes text,
  is_flagged boolean default false,
  created_at timestamp with time zone default timezone('utc', now())
);

-- Measurements
create table measurements (
  id uuid primary key default gen_random_uuid(),
  customer_id uuid references customers(id) on delete cascade,
  name text not null,
  value text,
  unit text
);

-- CustomerFlags
create table customer_flags (
  id uuid primary key default gen_random_uuid(),
  customer_id uuid references customers(id) on delete cascade,
  vendor_id uuid references vendors(id) on delete cascade,
  reason text,
  created_at timestamp with time zone default timezone('utc', now())
);


create table inventory_items (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  item_name text not null,
  quantity numeric default 0,
  unit text,
  color text,
  notes text,
  last_updated_at timestamp with time zone default timezone('utc', now())
);


create table shopping_lists (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  name text,
  notes text,
  created_at timestamp with time zone default timezone('utc', now())
);

create table shopping_list_items (
  id uuid primary key default gen_random_uuid(),
  shopping_list_id uuid references shopping_lists(id) on delete cascade,
  item_name text not null,
  desired_quantity numeric,
  unit text,
  notes text
);


create table invoices (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  customer_id uuid references customers(id),
  total_amount numeric,
  status text default 'draft',
  payment_link text,
  payment_provider text,
  created_at timestamp with time zone default timezone('utc', now())
);

create table invoice_items (
  id uuid primary key default gen_random_uuid(),
  invoice_id uuid references invoices(id) on delete cascade,
  name text not null,
  quantity numeric default 1,
  unit_price numeric,
  notes text
);

create table invoice_attachments (
  id uuid primary key default gen_random_uuid(),
  invoice_id uuid references invoices(id) on delete cascade,
  file_url text not null,
  type text
);


create table payments (
  id uuid primary key default gen_random_uuid(),
  invoice_id uuid references invoices(id) on delete cascade,
  vendor_id uuid references vendors(id) on delete cascade,
  amount numeric,
  provider text,
  provider_transaction_id text,
  status text,
  received_at timestamp with time zone
);

create table kyc (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  document_urls jsonb,
  status text default 'pending',
  reviewed_by uuid,
  reviewed_at timestamp with time zone
);


-- Contributors
create table contributors (
  vendor_id uuid primary key references vendors(id) on delete cascade,
  is_contributor boolean default false,
  contributor_level text default 'basic',
  status text default 'active',
  discount_percentage numeric default 0,
  contribution_requirement integer default 0,
  current_month_contributions integer default 0,
  last_review_date date
);

-- Markets
create table markets (
  id uuid primary key default gen_random_uuid(),
  name text not null,
  city text,
  description text
);

-- Items with synonyms
create table items (
  id uuid primary key default gen_random_uuid(),
  canonical_name text not null,
  description text,
  synonyms text[]
);

-- Price Reports
create table price_reports (
  id uuid primary key default gen_random_uuid(),
  item_id uuid references items(id) on delete cascade,
  market_id uuid references markets(id) on delete cascade,
  vendor_id uuid references vendors(id) on delete cascade,
  price numeric,
  unit text,
  notes text,
  created_at timestamp with time zone default timezone('utc', now())
);


create table referrals (
  id uuid primary key default gen_random_uuid(),
  referrer_vendor_id uuid references vendors(id) on delete cascade,
  referred_vendor_id uuid references vendors(id) on delete cascade,
  status text default 'pending',
  bonus_amount numeric default 0,
  date timestamp with time zone default timezone('utc', now())
);


create table notifications (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id),
  title text not null,
  message text not null,
  sent_at timestamp with time zone default timezone('utc', now()),
  is_read boolean default false
);

create table greetings (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id),
  message text not null,
  time_of_day text,
  created_at timestamp with time zone default timezone('utc', now())
);


create table sellers (
  id uuid primary key default gen_random_uuid(),
  market_id uuid references markets(id),
  shop_name text not null,
  contact_info text,
  delivery_options text,
  approved_status text default 'pending'
);

create table seller_inventory (
  id uuid primary key default gen_random_uuid(),
  seller_id uuid references sellers(id) on delete cascade,
  item_id uuid references items(id) on delete cascade,
  price numeric,
  stock_level numeric,
  last_updated timestamp with time zone default timezone('utc', now())
);

create table orders (
  id uuid primary key default gen_random_uuid(),
  buyer_vendor_id uuid references vendors(id) on delete cascade,
  seller_id uuid references sellers(id) on delete cascade,
  shopping_list_id uuid references shopping_lists(id),
  status text default 'pending',
  total_price numeric
);

create table order_items (
  id uuid primary key default gen_random_uuid(),
  order_id uuid references orders(id) on delete cascade,
  item_id uuid references items(id),
  quantity numeric,
  unit_price numeric
);


-- ✅ Add customer_flags table to track globally flagged customers
create table customer_flags (
  id uuid primary key default gen_random_uuid(),
  customer_id uuid references customers(id) on delete cascade,
  flagged_by_vendor_id uuid references vendors(id) on delete cascade,
  reason text not null,
  additional_notes text,
  proof_url text, -- optional link to image, screenshot, etc
  status text default 'pending', -- 'pending', 'approved', 'rejected'
  created_at timestamptz default now()
);

create table reminders (
  id uuid primary key default gen_random_uuid(),
  vendor_id uuid references vendors(id) on delete cascade,
  customer_id uuid references customers(id) on delete cascade,
  message text not null,
  type text,
  generated_by_ai boolean default false,
  status text default 'draft',
  created_at timestamp with time zone default timezone('utc', now()),
  scheduled_at timestamp with time zone
);


-- Example RLS policy for vendors
create policy "Allow vendor access to their own data" 
  on customers
  for all
  using (vendor_id = auth.uid());
